/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.util;

import htsjdk.samtools.FileTruncatedException;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.seekablestream.SeekableBufferedStream;
import htsjdk.samtools.seekablestream.SeekableFileStream;
import htsjdk.samtools.seekablestream.SeekableHTTPStream;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.util.BlockCompressedFilePointerUtil;
import htsjdk.samtools.util.BlockCompressedStreamConstants;
import htsjdk.samtools.util.BlockGunzipper;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.LocationAware;
import htsjdk.samtools.util.RuntimeIOException;
import htsjdk.samtools.util.zip.InflaterFactory;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;

public class BlockCompressedInputStream
extends InputStream
implements LocationAware {
    public static final String INCORRECT_HEADER_SIZE_MSG = "Incorrect header size for file: ";
    public static final String UNEXPECTED_BLOCK_LENGTH_MSG = "Unexpected compressed block length: ";
    public static final String PREMATURE_END_MSG = "Premature end of file: ";
    public static final String CANNOT_SEEK_STREAM_MSG = "Cannot seek a position for a non-file stream";
    public static final String CANNOT_SEEK_CLOSED_STREAM_MSG = "Cannot seek a position for a closed stream";
    public static final String INVALID_FILE_PTR_MSG = "Invalid file pointer: ";
    private InputStream mStream = null;
    private boolean mIsClosed = false;
    private SeekableStream mFile = null;
    private byte[] mFileBuffer = null;
    private DecompressedBlock mCurrentBlock = null;
    private int mCurrentOffset = 0;
    private long mStreamOffset = 0L;
    private final BlockGunzipper blockGunzipper;
    private volatile ByteArrayOutputStream buf = null;
    private static final byte eol = 10;
    private static final byte eolCr = 13;

    public BlockCompressedInputStream(InputStream stream) {
        this(stream, true, BlockGunzipper.getDefaultInflaterFactory());
    }

    public BlockCompressedInputStream(InputStream stream, InflaterFactory inflaterFactory) {
        this(stream, true, inflaterFactory);
    }

    public BlockCompressedInputStream(InputStream stream, boolean allowBuffering) {
        this(stream, allowBuffering, BlockGunzipper.getDefaultInflaterFactory());
    }

    public BlockCompressedInputStream(InputStream stream, boolean allowBuffering, InflaterFactory inflaterFactory) {
        this.mStream = allowBuffering ? IOUtil.toBufferedStream(stream) : stream;
        this.mFile = null;
        this.blockGunzipper = new BlockGunzipper(inflaterFactory);
    }

    public BlockCompressedInputStream(File file) throws IOException {
        this(file, BlockGunzipper.getDefaultInflaterFactory());
    }

    public BlockCompressedInputStream(File file, InflaterFactory inflaterFactory) throws IOException {
        this.mFile = new SeekableFileStream(file);
        this.mStream = null;
        this.blockGunzipper = new BlockGunzipper(inflaterFactory);
    }

    public BlockCompressedInputStream(URL url) {
        this(url, BlockGunzipper.getDefaultInflaterFactory());
    }

    public BlockCompressedInputStream(URL url, InflaterFactory inflaterFactory) {
        this.mFile = new SeekableBufferedStream(new SeekableHTTPStream(url));
        this.mStream = null;
        this.blockGunzipper = new BlockGunzipper(inflaterFactory);
    }

    public BlockCompressedInputStream(SeekableStream strm) {
        this(strm, BlockGunzipper.getDefaultInflaterFactory());
    }

    public BlockCompressedInputStream(SeekableStream strm, InflaterFactory inflaterFactory) {
        this.mFile = strm;
        this.mStream = null;
        this.blockGunzipper = new BlockGunzipper(inflaterFactory);
    }

    public void setCheckCrcs(boolean check) {
        this.blockGunzipper.setCheckCrcs(check);
    }

    @Override
    public int available() throws IOException {
        if (this.mCurrentBlock == null || this.mCurrentOffset == this.mCurrentBlock.mBlock.length) {
            this.readBlock();
        }
        if (this.mCurrentBlock == null) {
            return 0;
        }
        return this.mCurrentBlock.mBlock.length - this.mCurrentOffset;
    }

    public boolean endOfBlock() {
        return this.mCurrentBlock != null && this.mCurrentOffset == this.mCurrentBlock.mBlock.length;
    }

    @Override
    public void close() throws IOException {
        if (this.mFile != null) {
            this.mFile.close();
            this.mFile = null;
        } else if (this.mStream != null) {
            this.mStream.close();
            this.mStream = null;
        }
        this.mFileBuffer = null;
        this.mCurrentBlock = null;
        this.mIsClosed = true;
    }

    @Override
    public int read() throws IOException {
        return this.available() > 0 ? this.mCurrentBlock.mBlock[this.mCurrentOffset++] & 0xFF : -1;
    }

    @Override
    public int read(byte[] buffer) throws IOException {
        return this.read(buffer, 0, buffer.length);
    }

    public String readLine() throws IOException {
        int available = this.available();
        if (available == 0) {
            return null;
        }
        if (null == this.buf) {
            this.buf = new ByteArrayOutputStream(8192);
        }
        this.buf.reset();
        boolean done = false;
        boolean foundCr = false;
        while (!done) {
            int linetmpPos = this.mCurrentOffset;
            int bCnt = 0;
            while (available-- > 0) {
                byte c = this.mCurrentBlock.mBlock[linetmpPos++];
                if (c == 10) {
                    done = true;
                    break;
                }
                if (foundCr) {
                    --linetmpPos;
                    done = true;
                    break;
                }
                if (c == 13) {
                    foundCr = true;
                    continue;
                }
                ++bCnt;
            }
            if (this.mCurrentOffset < linetmpPos) {
                this.buf.write(this.mCurrentBlock.mBlock, this.mCurrentOffset, bCnt);
                this.mCurrentOffset = linetmpPos;
            }
            if ((available = this.available()) != 0) continue;
            done = true;
        }
        return this.buf.toString();
    }

    @Override
    public int read(byte[] buffer, int offset, int length) throws IOException {
        int originalLength = length;
        while (length > 0) {
            int available = this.available();
            if (available == 0) {
                if (originalLength != length) break;
                return -1;
            }
            int copyLength = Math.min(length, available);
            System.arraycopy(this.mCurrentBlock.mBlock, this.mCurrentOffset, buffer, offset, copyLength);
            this.mCurrentOffset += copyLength;
            offset += copyLength;
            length -= copyLength;
        }
        return originalLength - length;
    }

    public void seek(long pos) throws IOException {
        int available;
        if (this.mIsClosed) {
            throw new IOException(CANNOT_SEEK_CLOSED_STREAM_MSG);
        }
        if (this.mFile == null) {
            throw new IOException(CANNOT_SEEK_STREAM_MSG);
        }
        long compressedOffset = BlockCompressedFilePointerUtil.getBlockAddress(pos);
        int uncompressedOffset = BlockCompressedFilePointerUtil.getBlockOffset(pos);
        if (this.mCurrentBlock != null && this.mCurrentBlock.mBlockAddress == compressedOffset) {
            available = this.mCurrentBlock.mBlock.length;
        } else {
            this.prepareForSeek();
            this.mFile.seek(compressedOffset);
            this.mStreamOffset = compressedOffset;
            this.mCurrentBlock = this.nextBlock(this.getBufferForReuse(this.mCurrentBlock));
            this.mCurrentOffset = 0;
            available = this.available();
        }
        if (uncompressedOffset > available || uncompressedOffset == available && !this.eof()) {
            throw new IOException(INVALID_FILE_PTR_MSG + pos + " for " + this.getSource());
        }
        this.mCurrentOffset = uncompressedOffset;
    }

    protected void prepareForSeek() {
    }

    private boolean eof() throws IOException {
        if (this.mFile.eof()) {
            return true;
        }
        return this.mFile.length() - (this.mCurrentBlock.mBlockAddress + (long)this.mCurrentBlock.mBlockCompressedSize) == (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length;
    }

    public long getFilePointer() {
        if (this.mCurrentBlock == null) {
            return BlockCompressedFilePointerUtil.makeFilePointer(0L, 0);
        }
        if (this.mCurrentOffset > 0 && this.mCurrentOffset == this.mCurrentBlock.mBlock.length) {
            return BlockCompressedFilePointerUtil.makeFilePointer(this.mCurrentBlock.mBlockAddress + (long)this.mCurrentBlock.mBlockCompressedSize, 0);
        }
        return BlockCompressedFilePointerUtil.makeFilePointer(this.mCurrentBlock.mBlockAddress, this.mCurrentOffset);
    }

    @Override
    public long getPosition() {
        return this.getFilePointer();
    }

    public static long getFileBlock(long bgzfOffset) {
        return BlockCompressedFilePointerUtil.getBlockAddress(bgzfOffset);
    }

    public static boolean isValidFile(InputStream stream) throws IOException {
        if (!stream.markSupported()) {
            throw new RuntimeException("Cannot test non-buffered stream");
        }
        stream.mark(18);
        byte[] buffer = new byte[18];
        int count = BlockCompressedInputStream.readBytes(stream, buffer, 0, 18);
        stream.reset();
        return count == 18 && BlockCompressedInputStream.isValidBlockHeader(buffer);
    }

    private static boolean isValidBlockHeader(byte[] buffer) {
        return buffer[0] == 31 && (buffer[1] & 0xFF) == 139 && (buffer[3] & 4) != 0 && buffer[10] == 6 && buffer[12] == 66 && buffer[13] == 67;
    }

    private void readBlock() throws IOException {
        this.mCurrentBlock = this.nextBlock(this.getBufferForReuse(this.mCurrentBlock));
        this.mCurrentOffset = 0;
        this.checkAndRethrowDecompressionException();
    }

    protected DecompressedBlock nextBlock(byte[] bufferAvailableForReuse) {
        return this.processNextBlock(bufferAvailableForReuse);
    }

    private void checkAndRethrowDecompressionException() throws IOException {
        if (this.mCurrentBlock.mException != null) {
            if (this.mCurrentBlock.mException instanceof IOException) {
                throw (IOException)this.mCurrentBlock.mException;
            }
            if (this.mCurrentBlock.mException instanceof RuntimeException) {
                throw (RuntimeException)this.mCurrentBlock.mException;
            }
            throw new RuntimeException(this.mCurrentBlock.mException);
        }
    }

    private byte[] getBufferForReuse(DecompressedBlock block) {
        if (block == null) {
            return null;
        }
        return block.mBlock;
    }

    protected DecompressedBlock processNextBlock(byte[] bufferAvailableForReuse) {
        if (this.mFileBuffer == null) {
            this.mFileBuffer = new byte[65536];
        }
        long blockAddress = this.mStreamOffset;
        try {
            int headerByteCount = this.readBytes(this.mFileBuffer, 0, 18);
            this.mStreamOffset += (long)headerByteCount;
            if (headerByteCount == 0) {
                return new DecompressedBlock(blockAddress, new byte[0], 0);
            }
            if (headerByteCount != 18) {
                return new DecompressedBlock(blockAddress, headerByteCount, new IOException(INCORRECT_HEADER_SIZE_MSG + this.getSource()));
            }
            int blockLength = this.unpackInt16(this.mFileBuffer, 16) + 1;
            if (blockLength < 18 || blockLength > this.mFileBuffer.length) {
                return new DecompressedBlock(blockAddress, blockLength, new IOException(UNEXPECTED_BLOCK_LENGTH_MSG + blockLength + " for " + this.getSource()));
            }
            int remaining = blockLength - 18;
            int dataByteCount = this.readBytes(this.mFileBuffer, 18, remaining);
            this.mStreamOffset += (long)dataByteCount;
            if (dataByteCount != remaining) {
                return new DecompressedBlock(blockAddress, blockLength, new FileTruncatedException(PREMATURE_END_MSG + this.getSource()));
            }
            byte[] decompressed = this.inflateBlock(this.mFileBuffer, blockLength, bufferAvailableForReuse);
            return new DecompressedBlock(blockAddress, decompressed, blockLength);
        }
        catch (IOException e) {
            return new DecompressedBlock(blockAddress, 0, e);
        }
    }

    private byte[] inflateBlock(byte[] compressedBlock, int compressedLength, byte[] bufferAvailableForReuse) throws IOException {
        int uncompressedLength = this.unpackInt32(compressedBlock, compressedLength - 4);
        if (uncompressedLength < 0) {
            throw new RuntimeIOException(this.getSource() + " has invalid uncompressedLength: " + uncompressedLength);
        }
        byte[] buffer = bufferAvailableForReuse;
        if (buffer == null || uncompressedLength != buffer.length) {
            buffer = new byte[uncompressedLength];
        }
        this.blockGunzipper.unzipBlock(buffer, compressedBlock, compressedLength);
        return buffer;
    }

    private String getSource() {
        return this.mFile == null ? "data stream" : this.mFile.getSource();
    }

    private int readBytes(byte[] buffer, int offset, int length) throws IOException {
        if (this.mFile != null) {
            return BlockCompressedInputStream.readBytes(this.mFile, buffer, offset, length);
        }
        if (this.mStream != null) {
            return BlockCompressedInputStream.readBytes(this.mStream, buffer, offset, length);
        }
        return 0;
    }

    private static int readBytes(SeekableStream file, byte[] buffer, int offset, int length) throws IOException {
        int bytesRead;
        int count;
        for (bytesRead = 0; bytesRead < length && (count = file.read(buffer, offset + bytesRead, length - bytesRead)) > 0; bytesRead += count) {
        }
        return bytesRead;
    }

    private static int readBytes(InputStream stream, byte[] buffer, int offset, int length) throws IOException {
        int bytesRead;
        int count;
        for (bytesRead = 0; bytesRead < length && (count = stream.read(buffer, offset + bytesRead, length - bytesRead)) > 0; bytesRead += count) {
        }
        return bytesRead;
    }

    private int unpackInt16(byte[] buffer, int offset) {
        return buffer[offset] & 0xFF | (buffer[offset + 1] & 0xFF) << 8;
    }

    private int unpackInt32(byte[] buffer, int offset) {
        return buffer[offset] & 0xFF | (buffer[offset + 1] & 0xFF) << 8 | (buffer[offset + 2] & 0xFF) << 16 | (buffer[offset + 3] & 0xFF) << 24;
    }

    public static FileTermination checkTermination(File file) throws IOException {
        return BlockCompressedInputStream.checkTermination(IOUtil.toPath(file));
    }

    public static FileTermination checkTermination(Path path) throws IOException {
        try (SeekableByteChannel channel = Files.newByteChannel(path, StandardOpenOption.READ);){
            FileTermination fileTermination = BlockCompressedInputStream.checkTermination(channel);
            return fileTermination;
        }
    }

    public static FileTermination checkTermination(SeekableByteChannel channel) throws IOException {
        long fileSize = channel.size();
        if (fileSize < (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length) {
            return FileTermination.DEFECTIVE;
        }
        long initialPosition = channel.position();
        boolean exceptionThrown = false;
        try {
            channel.position(fileSize - (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length);
            ByteBuffer lastBlockBuffer = ByteBuffer.allocate(BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length);
            BlockCompressedInputStream.readFully(channel, lastBlockBuffer);
            if (Arrays.equals(lastBlockBuffer.array(), BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK)) {
                FileTermination fileTermination = FileTermination.HAS_TERMINATOR_BLOCK;
                return fileTermination;
            }
            int bufsize = (int)Math.min(fileSize, 65536L);
            byte[] bufferArray = new byte[bufsize];
            channel.position(fileSize - (long)bufsize);
            BlockCompressedInputStream.readFully(channel, ByteBuffer.wrap(bufferArray));
            for (int i2 = bufferArray.length - BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length; i2 >= 0; --i2) {
                if (!BlockCompressedInputStream.preambleEqual(BlockCompressedStreamConstants.GZIP_BLOCK_PREAMBLE, bufferArray, i2, BlockCompressedStreamConstants.GZIP_BLOCK_PREAMBLE.length)) continue;
                ByteBuffer byteBuffer = ByteBuffer.wrap(bufferArray, i2 + BlockCompressedStreamConstants.GZIP_BLOCK_PREAMBLE.length, 4);
                byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
                int totalBlockSizeMinusOne = byteBuffer.getShort() & 0xFFFF;
                if (bufferArray.length - i2 == totalBlockSizeMinusOne + 1) {
                    FileTermination fileTermination = FileTermination.HAS_HEALTHY_LAST_BLOCK;
                    return fileTermination;
                }
                FileTermination fileTermination = FileTermination.DEFECTIVE;
                return fileTermination;
            }
            FileTermination fileTermination = FileTermination.DEFECTIVE;
            return fileTermination;
        }
        catch (Throwable e) {
            exceptionThrown = true;
            throw e;
        }
        finally {
            if (!exceptionThrown) {
                channel.position(initialPosition);
            }
        }
    }

    static void readFully(SeekableByteChannel channel, ByteBuffer dst) throws IOException {
        int bytesRead;
        int capacity = dst.capacity();
        for (int totalBytesRead = 0; totalBytesRead < capacity; totalBytesRead += bytesRead) {
            bytesRead = channel.read(dst);
            if (bytesRead != -1) continue;
            throw new EOFException();
        }
    }

    public static void assertNonDefectiveFile(File file) throws IOException {
        if (BlockCompressedInputStream.checkTermination(file) == FileTermination.DEFECTIVE) {
            throw new SAMException(file.getAbsolutePath() + " does not have a valid GZIP block at the end of the file.");
        }
    }

    private static boolean preambleEqual(byte[] preamble, byte[] buf, int startOffset, int length) {
        for (int i2 = 0; i2 < length; ++i2) {
            if (preamble[i2] == buf[i2 + startOffset]) continue;
            return false;
        }
        return true;
    }

    protected static class DecompressedBlock {
        private final byte[] mBlock;
        private final int mBlockCompressedSize;
        private final long mBlockAddress;
        private final Exception mException;

        public DecompressedBlock(long blockAddress, byte[] block, int compressedSize) {
            this.mBlock = block;
            this.mBlockAddress = blockAddress;
            this.mBlockCompressedSize = compressedSize;
            this.mException = null;
        }

        public DecompressedBlock(long blockAddress, int compressedSize, Exception exception) {
            this.mBlock = new byte[0];
            this.mBlockAddress = blockAddress;
            this.mBlockCompressedSize = compressedSize;
            this.mException = exception;
        }
    }

    public static enum FileTermination {
        HAS_TERMINATOR_BLOCK,
        HAS_HEALTHY_LAST_BLOCK,
        DEFECTIVE;

    }
}

